/**  @file bta_pixel_intrpl.c
*  
*    @brief This file implements the pixel interpolation filter
*  
*    BLT_DISCLAIMER
*  
*    @author Alex Falkensteiner
*  
*    @cond svn
*  
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*  
*    @endcond
*/


#include "bta_pixel_intrpl.h"
#include <stdlib.h>
#include <string.h>

#ifndef BTA_EXCLUDE_FILTERS

// Local prototypes
static BTA_Status pixelIntrplApply(BTA_FltPixelIntrplInst *inst, BTA_Channel *channel);

BTA_Status BFLTpixelIntrplInit(BTA_FltPixelIntrplConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    int i;
    BTA_FltPixelIntrplInst *inst;
    if (!handle || !config) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0;
    inst = (BTA_FltPixelIntrplInst *)calloc(1, sizeof(BTA_FltPixelIntrplInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
    inst->infoEventInst = infoEventInst;
    inst->pxIndicesOut = (uint32_t *)malloc(config->pxCount * sizeof(uint32_t));
    if (!inst->pxIndicesOut) {
        free(inst);
        return BTA_StatusOutOfMemory;
    }
    memcpy(inst->pxIndicesOut, config->pxIndicesOut, config->pxCount * sizeof(uint32_t));
    inst->pxIndicesInLens = (uint16_t *)malloc(config->pxCount * sizeof(uint16_t));
    if (!inst->pxIndicesInLens) {
        free(inst->pxIndicesOut);
        free(inst);
        return BTA_StatusOutOfMemory;
    }
    memcpy(inst->pxIndicesInLens, config->pxIndicesInLens, config->pxCount * sizeof(uint16_t));
    inst->pxIndicesIn = (uint32_t **)malloc(config->pxCount * sizeof(uint32_t *));
    if (!inst->pxIndicesIn) {
        free(inst->pxIndicesInLens);
        free(inst->pxIndicesOut);
        free(inst);
        return BTA_StatusOutOfMemory;
    }
    inst->pxCount = config->pxCount;
    for (i = 0; i < config->pxCount; i++) {
        inst->pxIndicesIn[i] = (uint32_t *)malloc(config->pxIndicesInLens[i] * sizeof(uint32_t));
        if (!inst->pxIndicesIn[i]) {
            for (i--; i >= 0; i--) {
                free(inst->pxIndicesIn[i]);
            }
            free(inst->pxIndicesInLens);
            free(inst->pxIndicesOut);
            free(inst->pxIndicesIn);
            free(inst);
            return BTA_StatusOutOfMemory;
        }
        memcpy(inst->pxIndicesIn[i], config->pxIndicesIn[i], config->pxIndicesInLens[i] * sizeof(uint32_t));
    }
    *handle = inst;
    return BTA_StatusOk;
}


BTA_Status BFLTpixelIntrplClose(BTA_FltHandle *handle) {
    int i;
    BTA_FltPixelIntrplInst **inst = (BTA_FltPixelIntrplInst **)handle;
    for (i = 0; i < (*inst)->pxCount; i++) {
        free((*inst)->pxIndicesIn[i]);
        (*inst)->pxIndicesIn[i] = 0;
    }
    free((*inst)->pxIndicesOut);
    (*inst)->pxIndicesOut = 0;
    free((*inst)->pxIndicesInLens);
    (*inst)->pxIndicesInLens = 0;
    free(*inst);
    *inst = 0;
    return BTA_StatusOk;
}


BTA_Status BFLTpixelIntrplApply(BTA_FltHandle handle, BTA_Frame **frame) {
    BTA_Status status;
    int chIn;
    if (!handle || !frame) {
        return BTA_StatusInvalidParameter;
    }
    if (!*frame) {
        return BTA_StatusInvalidParameter;
    }
    BTA_FltPixelIntrplInst *inst = (BTA_FltPixelIntrplInst *)handle;
    for (chIn = 0; chIn < (*frame)->channelsLen; chIn++) {
        status = pixelIntrplApply(inst, (*frame)->channels[chIn]);
        if (status != BTA_StatusOk) {
            return status;
        }
    }
    return BTA_StatusOk;
}


static BTA_Status pixelIntrplApply(BTA_FltPixelIntrplInst *inst, BTA_Channel *channel) {
    int i, j;
    int32_t sum;
    for (i = 0; i < inst->pxCount; i++) {
        sum = 0;
        for (j = 0; j < inst->pxIndicesInLens[i]; j++) {
            switch (channel->dataFormat) {
            case BTA_DataFormatUInt16:
                sum += ((uint16_t *)channel->data)[inst->pxIndicesIn[i][j]];
                break;
            default:
                break;
            }
        }
        switch (channel->dataFormat) {
        case BTA_DataFormatUInt16:
            ((uint16_t *)channel->data)[inst->pxIndicesOut[i]] = sum / inst->pxIndicesInLens[i];
            break;
        default:
            BTAinfoEventHelper(inst->infoEventInst, 10, BTA_StatusNotSupported, "BFLTpixelIntrplApply: unsupported format", 0);
            break;
        }
    }
    return BTA_StatusOk;
}

#endif